<?php
// synthetic_check.php
require __DIR__ . '/monitor_helpers.php';
date_default_timezone_set('Asia/Kolkata');

$SITE = getenv('SITE_URL') ?: 'https://shop.tagdaraho.in';
$PRODUCT_URL = getenv('TEST_PRODUCT_URL') ?: ''; // set this env var to the product you want tested
$STATE_FILE = rtrim(getenv('MONITOR_BASE') ?: '/home/monitor', '/') . '/journey_state.json';
$USER_AGENT = 'Mozilla/5.0 (compatible; Tagdaraho-Monitor/1.0; +https://shop.tagdaraho.in)';

function http_request($method, $url, $opts = []) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_HEADER, true);
    curl_setopt($ch, CURLOPT_USERAGENT, getenv('MONITOR_USER_AGENT') ?: $GLOBALS['USER_AGENT']);
    if (!empty($opts['headers'])) curl_setopt($ch, CURLOPT_HTTPHEADER, $opts['headers']);
    if (!empty($opts['cookies'])) curl_setopt($ch, CURLOPT_COOKIE, implode('; ', $opts['cookies']));
    if (strtoupper($method) === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $opts['postfields'] ?? '');
    }
    $res = curl_exec($ch);
    $err = curl_error($ch);
    $info = curl_getinfo($ch);
    curl_close($ch);
    if ($res === false) throw new Exception("HTTP $method $url failed: $err");
    $header_size = $info['header_size'] ?? 0;
    $hdr = substr($res, 0, $header_size);
    $body = substr($res, $header_size);
    return ['headers'=>$hdr, 'body'=>$body, 'info'=>$info];
}

function detect_gokwick($hay) {
    $patterns = ['gokwick', 'gok-wick', 'gokwickpay', 'gokwick.co', 'gokwick.in'];
    $lower = strtolower($hay);
    foreach ($patterns as $p) if (strpos($lower, $p) !== false) return true;
    return false;
}

try {
    if (empty($PRODUCT_URL)) {
        throw new Exception("TEST_PRODUCT_URL not set. Please export TEST_PRODUCT_URL and re-run.");
    }
    $product_url = $PRODUCT_URL;

    $res = http_request('GET', $product_url);
    $code = $res['info']['http_code'] ?? 0;
    if ($code >= 400) throw new Exception("Product page returned HTTP $code");
    $product_html = $res['body'];

    $variant_id = null;
    if (preg_match('/"id":\s*([0-9]{6,})/', $product_html, $m)) {
        $variant_id = $m[1];
    } elseif (preg_match('/name="id" value="(\d+)"/', $product_html, $m2)) {
        $variant_id = $m2[1];
    }
    if (!$variant_id) {
        if (preg_match('/var meta = (.*?);\n/s', $product_html, $m3)) {
            if (preg_match('/"id":\s*([0-9]{6,})/', $m3[1], $m4)) $variant_id = $m4[1];
        }
    }
    if (!$variant_id) {
        throw new Exception("Variant id not found on product page. Product may be JS-built.");
    }

    $add_url = rtrim(parse_url($product_url, PHP_URL_SCHEME) . '://' . parse_url($product_url, PHP_URL_HOST), '/') . '/cart/add';
    $post = http_build_query(['id' => $variant_id, 'quantity' => 1]);
    $resp = http_request('POST', $add_url, ['postfields' => $post, 'headers'=>['Content-Type: application/x-www-form-urlencoded']]);
    $code = $resp['info']['http_code'] ?? 0;
    if ($code >= 400) {
        throw new Exception("Add-to-cart POST returned HTTP $code");
    }

    $base = parse_url($product_url, PHP_URL_SCHEME) . '://' . parse_url($product_url, PHP_URL_HOST);
    $checkout_url = $base . '/checkout';
    $chk = http_request('GET', $checkout_url);
    $chk_body = $chk['body'] ?? '';
    $chk_code = $chk['info']['http_code'] ?? 0;
    if ($chk_code >= 400) {
        throw new Exception("Checkout page returned HTTP $chk_code");
    }

    $combined = $resp['body'] . "\n" . $chk_body;
    if (detect_gokwick($combined) || detect_gokwick($chk['info']['url'] ?? '')) {
        echo "Gokwick detected in flow. Journey OK at " . date('Y-m-d H:i:s') . PHP_EOL;
    } else {
        throw new Exception("Gokwick not detected. Checkout may use JS to redirect to payment gateway.");
    }

} catch (Exception $e) {
    $msg = "Synthetic checkout failure: " . $e->getMessage() . "\nTime: " . date('Y-m-d H:i:s');
    $state = [];
    if (file_exists($STATE_FILE)) $state = json_decode(file_get_contents($STATE_FILE), true);
    $last_alert = $state['last_alert'] ?? 0;
    $suppress_window = intval(getenv('ALERT_SUPPRESS_SECONDS') ?: 1800);
    if (time() - $last_alert > $suppress_window) {
        send_alert("[ALERT] Synthetic checkout failure for {$SITE}", $msg);
        $state['last_alert'] = time();
        file_put_contents($STATE_FILE, json_encode($state));
    } else {
        error_log("Alert suppressed (last sent within window).");
    }
    echo $msg . PHP_EOL;
}
?>
