<?php
// pagespeed_check.php
require __DIR__ . '/monitor_helpers.php';
date_default_timezone_set('Asia/Kolkata');

$API_KEY = getenv('PAGESPEED_API_KEY');
$SITE_URL = getenv('SITE_URL') ?: 'https://shop.tagdaraho.in';
$THRESHOLD_SCORE = intval(getenv('PAGESPEED_THRESHOLD') ?: 50);
$STATE_FILE = rtrim(getenv('MONITOR_BASE') ?: '/home/monitor', '/') . '/pagespeed_state.json';

function fetch_pagespeed($url, $strategy, $key) {
    $qs = http_build_query([
        'url' => $url,
        'strategy' => $strategy,
        'key' => $key,
        'category' => ['performance','accessibility','best-practices','seo']
    ]);
    $endpoint = "https://www.googleapis.com/pagespeedonline/v5/runPagespeed?$qs";
    $ch = curl_init($endpoint);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    if ($res === false) throw new Exception("Pagespeed request error: $err");
    return json_decode($res, true);
}

function get_score($resp) {
    return isset($resp['lighthouseResult']['categories']['performance']['score']) ? (int)round($resp['lighthouseResult']['categories']['performance']['score']*100) : null;
}

try {
    if (!$API_KEY) throw new Exception("PAGESPEED_API_KEY missing.");
    $strategies = ['mobile','desktop'];
    $results = [];
    foreach ($strategies as $s) {
        $resp = fetch_pagespeed($SITE_URL, $s, $API_KEY);
        $score = get_score($resp);
        $audits = $resp['lighthouseResult']['audits'] ?? [];
        $results[$s] = [
            'score' => $score,
            'fcp' => $audits['first-contentful-paint']['displayValue'] ?? 'n/a',
            'lcp' => $audits['largest-contentful-paint']['displayValue'] ?? 'n/a',
            'cls' => $audits['cumulative-layout-shift']['displayValue'] ?? 'n/a',
        ];
    }

    $prev = [];
    if (file_exists($STATE_FILE)) $prev = json_decode(file_get_contents($STATE_FILE), true);

    $alerts = [];
    foreach ($results as $k => $r) {
        $score = $r['score'] ?? null;
        if ($score === null) $alerts[] = "PageSpeed ($k) returned no score.";
        else {
            if ($score < $THRESHOLD_SCORE) $alerts[] = "Performance ($k) low: $score (threshold $THRESHOLD_SCORE)";
            $prevScore = $prev[$k]['score'] ?? null;
            if ($prevScore !== null && ($prevScore - $score) >= 15) $alerts[] = "Performance ($k) dropped sharply: was $prevScore now $score.";
        }
    }

    file_put_contents($STATE_FILE, json_encode(array_map(function($v){ return ['score'=>$v['score'] ?? 0]; }, $results)));

    if (!empty($alerts)) {
        $subject = "[ALERT] site speed issues for {$SITE_URL}";
        $body = "Detected speed issues for {$SITE_URL}\n\n" . implode("\n", $alerts) . "\n\nDetails:\n";
        foreach ($results as $s => $r) {
            $body .= strtoupper($s) . " - Score: " . ($r['score'] ?? 'n/a') . " | FCP: {$r['fcp']} | LCP: {$r['lcp']} | CLS: {$r['cls']}\n";
        }
        $body .= "\nTime: " . date('Y-m-d H:i:s') . "\n";
        send_alert($subject, $body);
    }

    echo "Pagespeed run complete. Alerts: " . count($alerts) . PHP_EOL;

} catch (Exception $e) {
    $msg = "Pagespeed checker error: " . $e->getMessage();
    send_alert("[ERROR] pagespeed checker failed", $msg);
    echo $msg . PHP_EOL;
}
?>
