<?php
// monitor_helpers.php
// Shared helper functions for Mailgun + Gupshup alerts and WhatsApp toggle.
// Place this file in the same folder as the other scripts.

date_default_timezone_set('Asia/Kolkata');

/**
 * Check whether WhatsApp alerts are enabled.
 *
 * Priority:
 *  1. MONITOR_CONFIG_FILE (JSON) : {"enable_whatsapp": true}
 *  2. ENABLE_WHATSAPP environment variable ("1"/"0", "true"/"false")
 *  3. Default: enabled (true) if GUPSHUP vars exist
 */
function is_whatsapp_enabled() {
    $cfgPath = getenv('MONITOR_CONFIG_FILE') ?: rtrim(getenv('MONITOR_BASE') ?: '/home/monitor', '/') . '/monitor_config.json';
    if ($cfgPath && file_exists($cfgPath)) {
        $json = @file_get_contents($cfgPath);
        if ($json !== false) {
            $data = json_decode($json, true);
            if (is_array($data) && array_key_exists('enable_whatsapp', $data)) {
                return (bool)$data['enable_whatsapp'];
            }
        }
    }

    $env = getenv('ENABLE_WHATSAPP');
    if ($env !== false && $env !== null) {
        $env = strtolower(trim($env));
        if (in_array($env, ['0','false','no','off'], true)) return false;
        if (in_array($env, ['1','true','yes','on'], true)) return true;
    }

    $apiKey = getenv('GUPSHUP_API_KEY');
    $to = getenv('ALERT_WHATSAPP_TO');
    $source = getenv('GUPSHUP_SOURCE');
    return (!empty($apiKey) && !empty($to) && !empty($source));
}

/**
 * Send email via Mailgun
 */
function mailgun_send($to, $subject, $text) {
    $apiKey = getenv('MAILGUN_API_KEY');
    $domain = getenv('MAILGUN_DOMAIN');
    $from = getenv('ALERT_EMAIL_FROM') ?: "monitor@{$domain}";

    if (!$apiKey || !$domain) {
        error_log("Mailgun not configured. Skipping email.");
        return false;
    }

    $url = "https://api.mailgun.net/v3/{$domain}/messages";
    $post = [
        'from' => $from,
        'to' => $to,
        'subject' => $subject,
        'text' => $text
    ];

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_USERPWD, 'api:' . $apiKey);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    $info = curl_getinfo($ch);
    curl_close($ch);
    if ($res === false) {
        error_log("Mailgun send error: $err");
        return false;
    }
    if (($info['http_code'] ?? 0) >= 400) {
        error_log("Mailgun returned HTTP {$info['http_code']}: $res");
        return false;
    }
    return true;
}

/**
 * Send WhatsApp via Gupshup (respects is_whatsapp_enabled())
 */
function gupshup_send_whatsapp($to_number, $message_text) {
    if (!is_whatsapp_enabled()) {
        error_log("WhatsApp disabled by config/env. Skipping Gupshup send.");
        return false;
    }

    $apiKey = getenv('GUPSHUP_API_KEY');
    $source = getenv('GUPSHUP_SOURCE'); // e.g. 9199XXXXXXXX
    if (!$apiKey || !$source || !$to_number) {
        error_log("Gupshup not configured properly. Skipping WhatsApp.");
        return false;
    }

    $url = "https://api.gupshup.io/sm/api/v1/msg";
    $payload = [
        "channel" => "whatsapp",
        "source"  => $source,
        "destination" => $to_number,
        "message" => [
            "type" => "text",
            "text" => $message_text
        ]
    ];
    $json = json_encode($payload);

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Content-Type: application/json",
        "apikey: {$apiKey}"
    ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    $info = curl_getinfo($ch);
    curl_close($ch);
    if ($res === false) {
        error_log("Gupshup send error: $err");
        return false;
    }
    if (($info['http_code'] ?? 0) >= 400) {
        error_log("Gupshup returned HTTP {$info['http_code']}: $res");
        return false;
    }
    return true;
}

/**
 * Unified alert helper — mail + optional WhatsApp (best effort)
 */
function send_alert($subject, $body) {
    $emailTo = getenv('ALERT_EMAIL_TO');
    $whTo = getenv('ALERT_WHATSAPP_TO');

    if ($emailTo) {
        mailgun_send($emailTo, $subject, $body);
    }
    if ($whTo) {
        gupshup_send_whatsapp($whTo, "{$subject}\n\n{$body}");
    }
}
?>
